/*****************************************************************************
 *
 * Copyright (c) ComponentOne, LLC.
 * Portions Copyright (c) 1999, KL GROUP INC.
 * http://www.componentone.com
 *
 * This file is provided for demonstration and educational uses only.
 * Permission to use, copy, modify and distribute this file for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of ComponentOne not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * ComponentOne.
 *
 * COMPONENTONE MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY
 * OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 * PURPOSE, OR NON-INFRINGEMENT. COMPONENTONE SHALL NOT BE LIABLE FOR ANY
 * DAMAGES SUFFERED BY USERS AS A RESULT OF USING, MODIFYING OR
 * DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 *
 *****************************************************************************/

/* panel.c */
#define STRICT
#include <windows.h>
#include "olch2d.h"
#include "oc_color.h"
#include "panel.h"
#include "resource.h"

/* Text Strings for the chart */
TCHAR *header_format = _T("(%d) Michelle's Microchips");
TCHAR header_string_buf[80];
TCHAR *header_strings[] = { header_string_buf, NULL };
TCHAR *footer_strings[] = { _T("1963 Quarterly Results"), NULL };

/* Label Strings */
TCHAR *set_labels[] = { _T("Expenses"), _T("Revenue"), NULL };
TCHAR *point_labels[] = { _T("Q1"), _T("Q2"), _T("Q3"), _T("Q4"), NULL };

static HXRT2D hChart;

void SetUpChart(HWND hWnd, XrtDataHandle my_data)
{
	static int counter = 1;
	wsprintf(header_string_buf, header_format, counter++);

	XrtSetValues(hChart,
		XRT_BORDER,                 XRT_BORDER_PLAIN,
		XRT_BORDER_WIDTH,           1,
		XRT_DATA, 					my_data,
		XRT_HEADER_STRINGS,			header_strings,
		XRT_FOOTER_STRINGS, 		footer_strings,
		XRT_SET_LABELS, 			set_labels,
		XRT_POINT_LABELS,			point_labels,
		XRT_XANNOTATION_METHOD,		XRT_ANNO_POINT_LABELS,
		XRT_GRAPH_BACKGROUND_COLOR, ocColorCornsilk,
		XRT_DATA_AREA_BACKGROUND_COLOR, ocColorGhostWhite,
		XRT_GRAPH_FOREGROUND_COLOR, XrtColorMaroon,
		XRT_GRAPH_DEPTH,            20,
		XRT_GRAPH_ROTATION,         20,
		XRT_GRAPH_INCLINATION,      20,
		NULL);

	CheckRadioButton(hWnd, IDBAR, IDPLOT, IDPLOT);
	CheckDlgButton(hWnd, IDC_3DFX, TRUE);
	CheckDlgButton(hWnd, IDC_ALARMZONES, FALSE);
	CheckDlgButton(hWnd, IDTRAN, FALSE);
	CheckDlgButton(hWnd, IDINV, FALSE);
}

LRESULT WINAPI
WndProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static HWND hwndChart;
	static XrtDataHandle my_data;
	static XrtDataHandle new_data;
	UINT wCheck;
	int i;

	switch (msg) {
    case XRTN_PALETTECHANGED:
        SendMessage(XrtGetWindow(hChart), WM_QUERYNEWPALETTE, 0, 0);
        break;

    case WM_QUERYNEWPALETTE:
    case WM_PALETTECHANGED:
        SendMessage(XrtGetWindow(hChart), msg, wParam, lParam);
        break;

	case WM_INITDIALOG:
		hwndChart = GetDlgItem(hWnd, IDCHART);
		hChart = XrtCreate();
		XrtAttachWindow(hChart, hwndChart);
		my_data = XrtDataCreateFromFile((LPTSTR) _T("mm63.dat"), NULL);

		SetUpChart(hWnd, my_data);

		SetFocus(hWnd);
		break;

	case WM_COMMAND:
		/* check which button was clicked and set chart properties appropriately */
        /* handle commands */
        switch(LOWORD(wParam)) {

		case IDM_EXIT:
			DestroyWindow(hWnd);
			break;

		case IDM_ABOUTDEMO:
		case IDMHELP:
			WinHelp(hWnd, _T("OLCH8DMO.HLP"), HELP_CONTEXT, 38);
			break;

		case IDM_ABOUTC1CHART:
			WinHelp(hWnd, _T("OLCH8DMO.HLP"), HELP_CONTEXT, 19);
			break;

		case IDBAR:
			/* change the chart to a bar chart */
			CheckRadioButton(hWnd, IDBAR, IDPLOT, (int)wParam);
			XrtSetValues(hChart, XRT_TYPE, XRT_TYPE_BAR, NULL);
			break;

		case IDPLOT:
			/* change the chart to a plot chart */
			CheckRadioButton(hWnd, IDBAR, IDPLOT, (int)wParam);
			XrtSetValues(hChart, XRT_TYPE, XRT_TYPE_PLOT, NULL);
			break;

		case IDC_ALARMZONES:
			wCheck = IsDlgButtonChecked(hWnd, IDC_ALARMZONES);
			XrtSetValues(hChart, XRT_REPAINT, FALSE, NULL);

			if(wCheck == BST_UNCHECKED)
			{
				XrtRemoveAllAlarmZones(hChart);
			}
			else
			{
				const int nZones = 4;
				COLORREF color;
				double upper, lower, inc;

				XrtGetValues(hChart,
					XRT_YAXIS_MIN, &lower,
					XRT_YAXIS_MAX, &upper,
					XRT_DATA_AREA_BACKGROUND_COLOR, &color,
					NULL);

				inc = (upper - lower) / nZones;

				for(i=0; i < nZones; i++)
				{
					TCHAR name[8];
					wsprintf(name, _T("%03d"), i);
					color -= 0x201810;
					XrtSetAlarmZone(hChart, name, TRUE,
						lower, lower + inc, -1, color, XRT_FPAT_SOLID);
					lower += inc;
				}
			}

			XrtSetValues(hChart, XRT_REPAINT, TRUE, NULL);
			break;

		case IDC_3DFX:
			wCheck = IsDlgButtonChecked(hWnd, IDC_3DFX);
			XrtSetValues(hChart, XRT_REPAINT, FALSE, NULL);

			if(wCheck == BST_UNCHECKED)
				i = 0;
			else
				i = 20;

			XrtSetValues(hChart,
				XRT_GRAPH_DEPTH,            i,
				XRT_GRAPH_ROTATION,         i,
				XRT_GRAPH_INCLINATION,      i,
				NULL);

			XrtSetValues(hChart, XRT_REPAINT, TRUE, NULL);
			break;

		case IDC_EMPTYCHART:
			wCheck = IsDlgButtonChecked(hWnd, IDC_EMPTYCHART);
		    XrtSetValues(hChart, XRT_REPAINT, FALSE, NULL);
			XrtReinitialize(hChart);
			if(wCheck == BST_UNCHECKED)
				SetUpChart(hWnd, my_data);
		    XrtSetValues(hChart, XRT_REPAINT, TRUE, NULL);
			break;

		case IDTRAN:
			/* transpose the points and sets of the chart */
			wCheck = IsDlgButtonChecked(hWnd, IDTRAN);
            new_data = XrtDataTranspose(my_data);
            if (new_data != NULL) {
		        XrtDataDestroy(my_data);
                my_data = new_data;
                if (wCheck == BST_UNCHECKED) {
		            XrtSetValues(hChart, XRT_DATA, my_data,
					             XRT_SET_LABELS, set_labels,
					             XRT_POINT_LABELS, point_labels,
                                 NULL);
                }
                else {
		            XrtSetValues(hChart, XRT_DATA, my_data,
					             XRT_SET_LABELS, point_labels,
					             XRT_POINT_LABELS, set_labels,
                                 NULL);
                }
                new_data = (XrtDataHandle) NULL;
            }
			break;

		case IDINV:
			/* invert the axes of the chart */
			wCheck = IsDlgButtonChecked(hWnd, IDINV);
			XrtSetValues(hChart, XRT_INVERT_ORIENTATION, wCheck, NULL);
			break;

		}
		break;

	case WM_SIZE:
		{
			int width;
			int height;
			RECT rect;

			width = LOWORD(lParam);
			height = HIWORD(lParam);

			GetWindowRect(hwndChart, &rect);
			SetWindowPos(hwndChart, NULL, rect.left, rect.top,
							width - 3, height - rect.top + 42, SWP_NOMOVE);
		}
		break;

	case WM_CLOSE:
		XrtDataDestroy(my_data);
		DestroyWindow(hWnd);
		break;

	case WM_DESTROY:
		PostQuitMessage(0);
		break;

	default:
		return FALSE;
	}
	return TRUE;
}


int PASCAL
_tWinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPTSTR lpszCmdLine, int nCmdShow)
{
	static TCHAR szAppName[] = _T("panel");
	HWND 		hWnd;
	MSG 		msg;
	WNDCLASS 	wc;
	DLGPROC 	dlgprc;
	HACCEL      hAccel;

	if (!hPrevInstance) {
		wc.style			= CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc		= WndProc;
		wc.cbClsExtra		= 0;
		wc.cbWndExtra		= 0;
		wc.hInstance		= hInstance;
		wc.hIcon 			= LoadIcon (hInstance, _T("olch2dicon"));
		wc.hCursor			= LoadCursor (NULL, IDC_ARROW);
		wc.hbrBackground	= GetStockObject (WHITE_BRUSH);
		wc.lpszMenuName		= _T("PanelMenu");
		wc.lpszClassName 	= szAppName;

		if (!RegisterClass(&wc)) return FALSE;
	}

	hAccel = LoadAccelerators(hInstance, _T("PanelAccelerators"));

	dlgprc = (DLGPROC) MakeProcInstance((FARPROC)WndProc, hInstance);
	hWnd = CreateDialog(hInstance, szAppName, 0, dlgprc);
	ShowWindow(hWnd, nCmdShow);

	while (GetMessage(&msg, NULL, 0, 0)) {
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			if (!IsDialogMessage(hWnd, &msg)) {
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}
	}

	// remember to free allocated memory
    XrtDestroy(hChart);

    return (int)msg.wParam;
}
